<?php
/**
 * @version	    1.0
 * @package		JPlugin
 * @subpackage	Authentication
 * @copyright	Scouting Nederland 2011, All rights reserved
 * @license		GNU/GPL, see LICENSE.php
 *
 * Replacement package for the OpenID plugin in Joomla 1.5.
 * Uses the PHP5 OpenID library of Jan-Rain {@link http://www.janrain.com/openid-enabled}
 */

// Check to ensure this file is included in Joomla!
defined('_JEXEC') or die('Restricted access');

/**
 * Determines the source of randomness.
 * Must be executed before the libary is loaded.
 */
function _initOpenIDRandomness() {
    if (!defined('Auth_OpenID_RAND_SOURCE')) {
        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            define('Auth_OpenID_RAND_SOURCE', null);
        } else {
            $f = @fopen('/dev/urandom', 'r');
            if ($f !== false) {
                define('Auth_OpenID_RAND_SOURCE', '/dev/urandom');
                fclose($f);
            } else {
                $f = @fopen('/dev/random', 'r');
                if ($f !== false) {
                    define('Auth_OpenID_RAND_SOURCE', '/dev/urandom');
                    fclose($f);
                } else {
                    define('Auth_OpenID_RAND_SOURCE', null);
                }
            }
        }
    }
}

/**
 * Includes the OpenID library files and adds the path to the include path
 */
function _includeOpenIDLibrary() {
    $path_extra = dirname(__FILE__) . '/openid/';
    $path = ini_get('include_path');
    $path = $path_extra . PATH_SEPARATOR . $path;
    ini_set('include_path', $path);
    require_once 'openid/Auth/OpenID/Consumer.php';
    require_once 'openid/Auth/OpenID/FileStore.php';
    require_once 'openid/Auth/OpenID/SReg.php';
    require_once 'openid/Auth/OpenID/URINorm.php';
    require_once 'openid/openidDatabaseConnection.php';
    require_once 'openid/openidThemeExtension.php';
    require_once 'openid/Auth/OpenID/MySQLStore.php';
    require_once 'openid/DB/Store/MySQL.php';
}

_initOpenIDRandomness();
_includeOpenIDLibrary();

jimport('joomla.plugin.plugin');
jimport('joomla.application.router');
jimport('joomla.log.log');

/**
 * OpenID Authentication Plugin
 * Provides OpenID OP authentication
 *
 * Class uses OpenID authentication like the OpenID Authentication plugin of Joomla 1.5.
 * However this class uses the PHP5 OpenID library provided by Jan-Rain {@link http://www.janrain.com/openid-enabled}
 *
 *
 * @author      Frits Zwegers
 * @package		JPlugin
 * @subpackage	Authentication
 */
class plgAuthenticationOpenid extends JPlugin {
    /**
     * Session namespace for OpenID variables
     * @var string
     */
    const SESSION_NAMESPACE = 'openid';

    /**
     * Session token for indication that an authentication request
     * was sent to the OpenID provider
     * @var string
     */
    const OPENID_RESPONSE_TOKEN = 'response_token';

    /**
     * Session token to indicate URL that will process the response from the OpenID Provider
     * @var string
     */
    const PROCESS_URL_TOKEN = 'return_url';

    /**
     * Session token to indicate URL that will be trusted by the OpenID Provider when authentication
     * is successful
     * @var string
     */
    const TRUST_URL_TOKEN = 'trust_url';

    /**
     * Indicates if the immediate method should be userd
     * @var boolean
     */
    private $_immediate = false;

    private $_idSelect = false;

    /**
     * @param object $subject The object to observe
     * @param array $config  An array that holds the plugin configuration
     */
    public function __construct(&$subject, $config) {
        parent::__construct($subject, $config);
    }

    /**
     * This method should handle any authentication and report back to the subject
     * @param array $credentials Array holding the user credentials
     * @param array $options Array of extra options (return, entry_url)
     * @param JAuthenticationResponse $response Authentication response object
     * @return boolean
     */
    public function onUserAuthenticate($credentials, $options, &$response) {
        if ((bool)$this->params->get('disable_admin') && JFactory::getApplication()->isAdmin()) {
            return false; //Not for administrator section
        }

        $consumer = $this->_initConsumer($response);
        if ($consumer === false) {
            return false;
        }
        $this->_initImmediate($options);
        $this->_initIdSelect($credentials);

        if (array_key_exists('username', $credentials)) {
            $credentials['username'] = $this->_filterUsername($credentials['username']);
        }

        if (!$this->_isOpenIDResponse()) {
            return $this->_authenticateWithProvider($consumer, $credentials, $options, $response);
        }
        return $this->_completeAuthentication($consumer, $response);
    }

    /**
     * Create a consumer object
     * @param JAuthenticationResponse $response
     * @return Auth_OpenID_Consumer|false
     */
    protected function _initConsumer($response) {
        $store = $this->_initStore($response);
        if ($store === false) {
            return false;
        }
        return new Auth_OpenID_Consumer($store);
    }

    /**
     * Creates a store for OpenID data
     * @param $response
     * @return Auth_OpenID_OpenIDStore|false
     */
    protected function _initStore($response) {
        if ($this->params->get('store_database') == 1) {
            return $this->_initDbStore($response);
        } else {
            return $this->_initFileStore($response);
        }
    }

    /**
     * Initializes database store
     * @param JAuthenticationResponse $response
     * @return Auth_OpenID_MySQLStore
     */
    protected function _initDbStore($response) {
        $connection = new openidDatabaseConnection(JFactory::getDBO());
        $store = new openid_DB_STORE_MySQL($connection, '#__oid_associations', '#__oid_nonces');

       if (!$store->createTables()) {

           $response->type = JAuthentication::STATUS_FAILURE;
           $response->error_message = "Could not create tables for OpenID authentication in the database.";
           return false;
       }

        $store->cleanup();
        return $store;
    }

    /**
     * Initializes file store
     * @param JAuthenticationResponse $response
     * @return Auth_OpenID_FileStore|false
     */
    protected function _initFileStore($response) {
        jimport('joomla.filesystem.folder');

        // Create and/or start using the data store
        $store_path = JPATH_ROOT . '/tmp/_joomla_openid_store';
        if (!JFolder::exists($store_path) && !JFolder::create($store_path)) {
            $response->type = JAuthentication::STATUS_FAILURE;
            $response->error_message = "Could not create the FileStore directory '$store_path'. " . " Please check the effective permissions.";
            return false;
        }

        // Create store object
        return new Auth_OpenID_FileStore($store_path);
    }

    /**
     * Sets the value for the immediate property
     * @param array $options
     */
    protected function _initImmediate($options) {
        $this->_immediate = array_key_exists('immediate', $options) && (bool)$options['immediate'];
    }

    /**
     * Sets a flag to show of identity select is used
     * @param array $credentials
     */
    protected function _initIdSelect($credentials) {
        $this->_idSelect = !array_key_exists('username', $credentials) || empty($credentials['username']);
    }

    /**
     * Creates an authentication request and sends it to the provider
     *
     * @param Auth_OpenID_Consumer $consumer
     * @param array                $credentials Array containing key 'username'
     * @param array                $options
     * @param JAuthenticationResponse            $response
     *
     * @return bool|false
     */
    protected function _authenticateWithProvider($consumer, $credentials, $options, $response) {
        if (!$this->_idSelect) {
            $auth_request = $this->_initAuthenticationRequest($consumer, $this->_formatOpenID($credentials['username']), $response);
        } else {
            //Perform identifier selection
            $auth_request = $this->_initAuthenticationRequestWithoutDiscovery($consumer, $response);
        }

        if ($auth_request === false) {
            $response->type = JAuthentication::STATUS_FAILURE;
            $response->error_message = "Cannot create an OpenID authentication request.";
            return false;
        }

        $this->_appendSimpleRegistrationRequest($auth_request);
        $this->_appendThemeExtension($auth_request);

        $base_url = $this->_getBaseURI($credentials, $options);
        $process_url = $this->_getProcessURI($base_url);
        $trust_url = $this->_getTrustURI($base_url);

        if ($auth_request->shouldSendRedirect()) {
            return $this->_redirectAuthenticationRequest($auth_request, $trust_url, $process_url);
        }
        return $this->_generateAuthenticationForm($auth_request, $trust_url, $process_url);
    }

    /**
     * Parses the $id in the prefix url that can be set in the settings
     * @param string $id
     * @return string
     */
    protected function _formatOpenID($id) {
        return sprintf($this->params->get('prefix_url'), $id);
    }

    /**
     * Filters the username to lowercase and replaces diacritical characters
     * @param string $username
     * @return string
     */
    protected function _filterUsername($username) {
        return strtolower(str_replace($this->_getSearchArray(), $this->_getReplaceArray(), mb_strtolower($username, mb_detect_encoding($username))));
    }

    /**
     * Characters to search for in username
     * @return array
     */
    private function _getSearchArray() {
        return array(
			'Š', 'š', 'Ð','Ž', 'ž', 'À', 'Á', 'Â', 'Ã', 'Ä',
            'Å', 'Æ', 'Ç', 'È', 'É', 'Ê', 'Ë', 'Ì', 'Í', 'Î',
            'Ï', 'Ñ', 'Ò', 'Ó', 'Ô', 'Õ', 'Ö', 'Ø', 'Ù', 'Ú',
            'Û', 'Ü', 'Ý', 'Þ', 'ß','à', 'á', 'â', 'ã', 'ä',
            'å', 'æ', 'ç', 'è', 'é', 'ê', 'ë', 'ì', 'í', 'î',
            'ï', 'ð', 'ñ', 'ò', 'ó', 'ô', 'õ', 'ö', 'ø', 'ù',
            'ú', 'û', 'ý', 'ý', 'þ', 'ÿ', 'ƒ', 'é', 'ü'
        );
    }

    /**
     * Replacement characterers for searched array
     * @return array
     */
    private function _getReplaceArray() {
        return array(
            'S', 's', 'Dj','Z', 'z', 'A', 'A', 'A', 'A', 'A',
            'A', 'A', 'C', 'E', 'E', 'E', 'E', 'I', 'I', 'I',
            'I', 'N', 'O', 'O', 'O', 'O', 'O', 'O', 'U', 'U',
            'U', 'U', 'Y', 'B', 'Ss','a', 'a', 'a', 'a', 'a',
            'a', 'a', 'c', 'e', 'e', 'e', 'e', 'i', 'i', 'i',
            'i', 'o', 'n', 'o', 'o', 'o', 'o', 'o', 'o', 'u',
            'u', 'u', 'y', 'y', 'b', 'y', 'f', 'e', 'u'
        );
    }

    /**
     * Initializes the Authentication request
     *
     * @param Auth_OpenID_Consumer $consumer
     * @param string               $id URL of the OpenID
     * @param JAuthenticationResponse            $response
     *
     * @return Auth_OpenID_AuthRequest|false
     */
    protected function _initAuthenticationRequest($consumer, $id, $response) {
        $result = $consumer->begin($id);
        if (!$result instanceof Auth_OpenID_AuthRequest) {
            $response->type = JAuthentication::STATUS_FAILURE;
            $response->error_message = 'Authentication error : could not connect to the openid server';
            return false;
        }
        return $result;
    }

    /**
     * Initializes the Authentication request without discovery
     *
     * @param Auth_OpenID_Consumer $consumer
     * @param JAuthenticationResponse            $response
     *
     * @return Auth_OpenID_AuthRequest|false
     */
    protected function _initAuthenticationRequestWithoutDiscovery($consumer, $response) {
        $result = $consumer->beginWithoutDiscovery(Auth_OpenID_ServiceEndpoint::fromOPEndpointURL($this->params->get('endpoint_url')));
        if (!$result instanceof Auth_OpenID_AuthRequest) {
            $response->type = JAuthentication::STATUS_FAILURE;
            $response->error_message = 'Authentication error : could not connect to the openid server';
            return false;
        }
        return $result;
    }

    /**
     * Appends a Simple Registration request to the Authentication Request
     * This request asks for extra data-fields.
     *
     * @param Auth_OpenID_AuthRequest $auth_request
     *
     * @return null
     */
    protected function _appendSimpleRegistrationRequest($auth_request) {
        $result = Auth_OpenID_SRegRequest::build(
        array ('email', 'fullname'),
        array ('language','timezone', 'nickname')
        );

        if ($result) {
            $auth_request->addExtension($result);
        }
        return $result;
    }

    /**
     * Appends the selected theme of the login box (if any)
     * @param Auth_OpenID_AuthRequest $auth_request
     */
    protected function _appendThemeExtension($auth_request) {
        $theme = JFactory::getApplication()->input->get('theme', false);
        if ($theme !== false && strlen($theme) > 0) {
           $auth_request->addExtension(new openidThemeExtension($theme));
        }
    }

    /**
     * Creates a base URI to return to
     * Sets a response token.
     * @param array $credentials Contains key 'username'
     * @param array $options Array that could contain key 'entry_url' and 'return'
     * @return JURI
     */
    protected function _getBaseURI($credentials, $options) {
        $result = JURI::getInstance(isset($options['entry_url']) ? $options['entry_url'] : JURI::base());
        $token = JSession::getFormToken();
        $result->setQuery(array_merge($result->getQuery(true), array('option' => 'com_openid', 'username' => $credentials['username'], $token  => 1)));
        JFactory::getSession()->set(self::OPENID_RESPONSE_TOKEN, $token, self::SESSION_NAMESPACE);
        JFactory::getSession()->set('return_user_url', base64_encode($options['return']), self::SESSION_NAMESPACE);
        return $result;
    }

    /**
     * Creates the process URL and saves it in the session using key {@link self::PROCESS_URL_TOKEN}
     * @param JURI $base_uri
     * @return string
     */
    protected function _getProcessURI($base_uri) {
        $result = $base_uri->toString();
        JFactory::getSession()->set(self::PROCESS_URL_TOKEN, $result, self::SESSION_NAMESPACE);
        return $result;
    }

    /**
     * Creates the URL to gain trust for and saves it in the session using key {@link self::TRUST_URL_TOKEN}
     * @param JURI $base_url
     * @return string
     */
    protected function _getTrustURI($base_url) {
        $result = $base_url->toString(array('path','host','port','scheme'));
        $result = substr($result, 0, strrpos($result, '/'));
        JFactory::getSession()->set(self::TRUST_URL_TOKEN, $result, self::SESSION_NAMESPACE);
        return $result;
    }

    /**
     * Returns if the response token has been set.
     * When the response token is set, the code assumes a request was send to a OpenID provider
     * and this is the (indirect) response to that request.
     * @return boolean
     */
    protected function _isOpenIDResponse() {
        $session = JFactory::getSession();
        $URI = JURI::getInstance();
        return $session->has(self::OPENID_RESPONSE_TOKEN, self::SESSION_NAMESPACE) && $URI->getVar($session->get(self::OPENID_RESPONSE_TOKEN, false, self::SESSION_NAMESPACE),false) == 1;
    }

    /**
     * Redirects the user to the OpenID provider for authentication
     * For OpenID 1, send a redirect
     * @param Auth_OpenID_AuthRequest $auth_request
     * @param string $trust_url
     * @param string $process_url
     * @return false Returns false to allow to test the login function
     */
    protected function _redirectAuthenticationRequest($auth_request, $trust_url, $process_url) {
        $redirect_url = $auth_request->redirectURL($trust_url, $process_url, $this->_immediate);

        // If the redirect URL can't be built, display an error
        if (Auth_OpenID::isFailure($redirect_url)) {
            JFactory::getApplication()->enqueueMessage("Could not redirect to server: " . $redirect_url->message, 'error');
            return false;
        }
        JFactory::getApplication()->redirect($redirect_url);
        return false;
    }

    /**
     * Outputs a form to allow authentication with the OpenID provider
     * For OpenID 2, use a Javascript
     * @param Auth_OpenID_AuthRequest $auth_request
     * @param string $trust_url
     * @param string $process_url
     * @return false Returns false to end the authentication
     */
    protected function _generateAuthenticationForm($auth_request, $trust_url, $process_url) {
        // Generate form markup and render it.
        $form_html = $auth_request->htmlMarkup($trust_url, $process_url, $this->_immediate, array('id' => 'openid_message'));
        $message = $auth_request->getMessage($trust_url, $process_url);
        JFactory::getSession()->set(self::PROCESS_URL_TOKEN, $message->getArg('http://specs.openid.net/auth/2.0', 'return_to', JFactory::getSession()->get(self::PROCESS_URL_TOKEN, null, self::SESSION_NAMESPACE)), self::SESSION_NAMESPACE);
        if (Auth_OpenID::isFailure($form_html)) {
            JFactory::getApplication()->enqueueMessage("Failed to create a OpenID authentication form. " . $form_html->message, 'error');
            return false;
        }

        $mainframe = JFactory::getApplication();
        $mainframe->setBody($form_html);
        echo $mainframe->toString($mainframe->get('gzip'));
        $mainframe->close();
        return false;
    }

    /**
     * Completes the authentication
     * Result is either success, cancel or failure
     *
     * @param Auth_OpenID_Consumer $consumer
     * @param JAuthenticationResponse            $response
     *
     * @return bool
     */
    protected function _completeAuthentication($consumer, $response) {
        $auth_result = $consumer->complete(JFactory::getSession()->get(self::PROCESS_URL_TOKEN, null, self::SESSION_NAMESPACE));
        switch ($auth_result->status) {
            case Auth_OpenID_SUCCESS :
                if (JDEBUG) {
                    JFactory::getApplication()->enqueueMessage('OpenID Authentication successful');
                }
                $this->_processSuccessStatus($auth_result, $response);
                $application = JFactory::getApplication();
                $returnUrl = JFactory::getSession()->get('return_user_url', null, self::SESSION_NAMESPACE);
                if ($returnUrl !== null) {
                    $returnUrl = base64_decode($returnUrl);
                }
                $application->setUserState('login_return_url', $returnUrl);
                break;
            case Auth_OpenID_CANCEL :
                if (JDEBUG) {
                    JFactory::getApplication()->enqueueMessage('OpenID Authentication cancelled');
                }
                $this->_processCancelStatus($response);
                break;
            case Auth_OpenID_SETUP_NEEDED:
                if (JDEBUG) {
                    JFactory::getApplication()->enqueueMessage('OpenID Authentication setup needed');
                }
                $this->_processSetupNeeded($auth_result);
                break;
            default;
            case Auth_OpenID_FAILURE :
                if (JDEBUG) {
                    JFactory::getApplication()->enqueueMessage('OpenID Authentication failed: ' . $auth_result->message);
                }

                $this->_processFailureStatus($response);
                break;
        }
        return true;
    }

    /**
     * Creates a SUCCESS response
     * Processes the authentication when OpenID signals a successful result
     * @param Auth_OpenID_ConsumerResponse $result
     * @param JAuthenticationResponse $response
     */
    protected function _processSuccessStatus($result, $response) {
        $sreg = $this->_getSimpleRegistrationRespone();
        $response->username = $result->getDisplayIdentifier();
        $response->status = JAuthentication::STATUS_SUCCESS;
        $response->error_message = '';
        if (!isset($sreg['email']) || $this->params->get('fake_email') == 1) {
            if (isset($sreg['nickname'])) {
                if (JDEBUG) {
                    JFactory::getApplication()->enqueueMessage('OpenID Authentication setting User fake e-mail based on nickname');
                }
                $response->email = $sreg['nickname'] . $this->params->get('email_suffix');
            } else {
                if (JDEBUG) {
                    JFactory::getApplication()->enqueueMessage('OpenID Authentication setting User fake e-mail based on username');
                }
                $response->email = substr($response->username, strrpos($response->username, '/') +1) . $this->params->get('email_suffix');
            }
        } else {
            if (JDEBUG) {
                JFactory::getApplication()->enqueueMessage('OpenID Authentication setting User e-mail');
            }
            $response->email = $sreg['email'];
        }
        $response->fullname = isset($sreg['fullname']) ? $sreg['fullname'] : $response->username;
        $response->language = isset($sreg['language']) ? $sreg['language'] : '';
        $response->timezone = isset($sreg['timezone']) ? $sreg['timezone'] : '';
    }

	/**
     * Retrieves the response of the Simple Registration request
     * @return array Containing the required and possibly the optional values.
     */
    protected function _getSimpleRegistrationRespone() {
        return Auth_OpenID_SRegResponse::extractResponse(Auth_OpenID_SRegRequest::build(
            array ('email', 'fullname'),
            array ('language','timezone', 'nickname')
        ), $this->_filterSimpleRegistrationData(JFactory::getApplication()->input->getArray()))->contents();
    }

    /**
     * Filters the Simple Registration data
     * @param array $data
     * @return array
     */
    protected function _filterSimpleRegistrationData($data) {
        $result = array();
        foreach ($data as $key => $value) {
            if (strpos($key, 'openid_sreg_') !== false) {
                $result[substr($key, 12)] = $value;
            }
        }
        return $result;
    }

    /**
     * Sets a CANCEL response
     * @param JAuthenticationResponse $response
     */
    protected function _processCancelStatus($response) {
        $response->status = JAuthentication::STATUS_FAILURE;
        $response->error_message = 'Authentication cancelled';
    }

    /**
     * Sets a FAILURE response
     * @param JAuthenticationResponse $response
     */
    protected function _processFailureStatus($response) {
        $response->status = JAuthentication::STATUS_FAILURE;
        $response->error_message = 'Authentication failed';
    }

    /**
     * Redirects the user to the setup url
     *
     * @param           $result
     */
    protected function _processSetupNeeded($result) {
        JFactory::getApplication()->redirect($result->setup_url);
    }
}