<?php
/**
 * @package		JPlugin
 * @subpackage	Authentication
 * @version 1.0
 * @copyright Scouting Nederland, 2011, All rights reserved
 * @license		GNU/GPL, see LICENSE.php
 */
defined('_JEXEC') or die('Restricted access');
require_once 'Auth/OpenID/DatabaseConnection.php';

/**
 *
 * Class that ports the Joomla Database Connection to the OpenID library
 * @author Frits Zwegers
 */
class openidDatabaseConnection extends Auth_OpenID_DatabaseConnection {

    /**
     * @var JDatabaseDriver
     */
    private $_db;

    /**
     * Flag indicating if autocommit is enabled
     * @var boolean
     */
    private $_autoCommit = true;

    /**
     * @param JDatabaseDriver $dbo
     */
    public function __construct($connection) {
        $this->_db = $connection;
    }

	/**
     * Sets auto-commit mode on this database connection.
     *
     * @param bool $mode True if auto-commit is to be used; false if
     * not.
     */
    function autoCommit($mode) {
        $this->_autoCommit = (bool)$mode;
    }

    /**
     * Run an SQL query with the specified parameters, if any.
     *
     * @param string $sql An SQL string with placeholders.  The
     * placeholders are assumed to be specific to the database engine
     * for this connection.
     *
     * @param array $params An array of parameters to insert into the
     * SQL string using this connection's escaping mechanism.
     *
     * @return mixed $result The result of calling this connection's
     * internal query function.  The type of result depends on the
     * underlying database engine.  This method is usually used when
     * the result of a query is not important, like a DDL query.
     */
    function query($sql, $params = array()) {
        try {
            if ($this->_query($sql, $params) === false) {
                return false;
            }
            return $this->_db->execute();
        } catch (RuntimeException $e) {
            return false;
        }
    }

    /**
     * Sets the query in the database
     * @param string $query
     * @param array $params
     */
    private function _query($query, $params) {
        $query = $this->_db->replacePrefix($query);
        if (count($params) == 0) {
            $this->_db->setQuery($query);
            return true;
        }

        $name = 'qry' . rand(1000, 9999);
        $this->_db->setQuery("PREPARE `{$name}` FROM '" . $query . "';");
        if ($this->_db->execute() === false) {
            return false;
        }
        $i = 0;
        $setParams = array();
        foreach ($params as $param) {
            $this->_db->setQuery("SET @{$i} = " . $this->_db->quote($param) . ";");
            if ($this->_db->execute() === false) {
                return false;
            }
            $setParams[] = "@{$i}";
            $i++;
        }
        $this->_db->setQuery("EXECUTE {$name} USING " . implode(', ', $setParams) . ";");

        return true;
    }

    /**
     * Starts a transaction on this connection, if supported.
     */
    function begin() {
        $this->_db->transactionStart();
    }

    /**
     * Commits a transaction on this connection, if supported.
     */
    function commit() {
        $this->_db->transactionCommit();
    }

    /**
     * Performs a rollback on this connection, if supported.
     */
    function rollback() {
        $this->_db->transactionRollback();
    }

    /**
     * Run an SQL query and return the first column of the first row
     * of the result set, if any.
     *
     * @param string $sql An SQL string with placeholders.  The
     * placeholders are assumed to be specific to the database engine
     * for this connection.
     *
     * @param array $params An array of parameters to insert into the
     * SQL string using this connection's escaping mechanism.
     *
     * @return mixed $result The value of the first column of the
     * first row of the result set.  False if no such result was
     * found.
     */
    function getOne($sql, $params = array()) {
        $result = $this->_query($sql, $params);
        if ($result === false) {
            return false;
        }
        $result = $this->_db->loadResult();
        if (empty($result)) {
            return false;
        }
        return $result;
    }

    /**
     * Run an SQL query and return the first row of the result set, if
     * any.
     *
     * @param string $sql An SQL string with placeholders.  The
     * placeholders are assumed to be specific to the database engine
     * for this connection.
     *
     * @param array $params An array of parameters to insert into the
     * SQL string using this connection's escaping mechanism.
     *
     * @return array $result The first row of the result set, if any,
     * keyed on column name.  False if no such result was found.
     */
    function getRow($sql, $params = array()) {
        $result = $this->_query($sql, $params);
        if ($result === false) {
            return false;
        }
        $result = $this->_db->loadAssoc();
        if (!is_array($result) || empty($result)) {
            return false;
        }
        return $result;
    }

    /**
     * Run an SQL query with the specified parameters, if any.
     *
     * @param string $sql An SQL string with placeholders.  The
     * placeholders are assumed to be specific to the database engine
     * for this connection.
     *
     * @param array $params An array of parameters to insert into the
     * SQL string using this connection's escaping mechanism.
     *
     * @return array $result An array of arrays representing the
     * result of the query; each array is keyed on column name.
     */
    function getAll($sql, $params = array()) {
        $result = $this->_query($sql, $params);
        if ($result === false) {
            return false;
        }
        $result = $this->_db->loadAssocList();
        if (!is_array($result) || empty($result)) {
            return false;
        }
        return $result;
    }

    function affectedRows() {
        return $this->_db->getAffectedRows();
    }
}